/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Text;
using System.Collections;
using System.Collections.Specialized;
using System.Runtime.Serialization;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Support;
using Borland.Eco.Exceptions;
using Borland.Eco.DataRepresentation;
using Borland.Eco.Interfaces;
using Borland.Eco.Persistence.ORMapping;
using Borland.Eco.Persistence.Configuration;
using Borland.Eco.Persistence.Connection;

namespace Borland.Eco.Persistence
{
	public class AttributeKeyMapper: GenericKeyMapper, IKeyMapper
	{
		public void IdToParameters(ObjectId id, IDataParameter[] parameters)
		{
			if (id == null)
			{
				for (int i=0; i<KeyMappers.Count; i++)
				GetKeyMapper(i).ValueToParameter(null, parameters[i]);
			}
			else if (KeyMappers.Count == 1)
				GetKeyMapper(0).ValueToParameter(id.Key, parameters[0]);
			else
			{
				ContentEqualArray idarr = id.Key as ContentEqualArray;
				for (int i=0; i<KeyMappers.Count; i++)
					GetKeyMapper(i).ValueToParameter(idarr.GetItem(i), parameters[i]);
			}
		}

		private System.Object GetFieldValue(int fieldno, IField[] keyFields)
		{
			IField f = keyFields[fieldno];
			if (f.IsNull)
				return null;
			else
				return GetKeyMapper(fieldno).ColumnToValue(f.Value);
		}

		public ObjectId IdFromFields(IField[] keyFields, ArrayList keyMappers, int classId, int timeStamp)
		{
			System.Object value;

			if (KeyMappers.Count == 1)
				value = GetFieldValue(0, keyFields);
			else
			{
				ContentEqualArray idarr = new ContentEqualArray(KeyMappers.Count);
				for (int i = 0; i<KeyMappers.Count; i++)
				{
					value = GetFieldValue(i, keyFields);
					if (value == null)
					{
						idarr = null;
						break;
					}
					idarr.SetItem(i, value);
				}
				value = idarr;
			}

			if (value == null)
				return null;
			else if (timeStamp != VersionConst.CurrentVersion)
				return new TimestampedDefaultId(value, classId, timeStamp);
			else
				return new DefaultId(value, classId);
		}

		public int ColumnCount()
		{
			return KeyMappers.Count;
		}

		public void ReserveId()
		{
			// Intentionally left blank
		}
		public ObjectId NewIdPreUpdate(IDatabase db, ObjectId oldId)
		{
			return null;
		}

		private System.Object GetMemberValue(System.Object member, IdTranslationList translationList, bool isOrderColumn)
		{
			SingleAssociationEnd singleLink = member as SingleAssociationEnd;
			if (singleLink != null)
			{
				if (singleLink.Id != null)
				{
					ObjectId newId = translationList.TranslateToNewId(singleLink.Id);
					if (!newId.IsStorable)
						throw new EcoException(InterfacesStringRes.sInternalIdEncountered);
					if (isOrderColumn)
						return singleLink.OrderNo;
					else
						return newId.Key;
				}
				else
					return null;
			}
			else
				return member;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="memberIndexes"/> is null</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="objContents"/> is null</exception>
		public ObjectId NewIdFromAttributes(ObjectContents objContents, ArrayList memberIndexes, IdTranslationList translationList)
		{
			if (memberIndexes == null) throw new ArgumentNullException("memberIndexes"); // Do not localize
			if (objContents == null) throw new ArgumentNullException("objContents"); // Do not localize
			if (memberIndexes.Count == 1)
			{
				MemberIndexColumnIndex memberIndexColIndex = memberIndexes[0] as MemberIndexColumnIndex;
				return new DefaultId(GetMemberValue(objContents.GetMember(memberIndexColIndex.MemberIndex), translationList, memberIndexColIndex.IsOrderColumn), objContents.ObjectId.ClassId);
			}
			else
			{
				ContentEqualArray idArr = new ContentEqualArray(memberIndexes.Count);
				for (int i=0; i<memberIndexes.Count; i++)
				{
					MemberIndexColumnIndex memberIndexColIndex = memberIndexes[i] as MemberIndexColumnIndex;

					idArr.SetItem(i, GetMemberValue(objContents.GetMember(memberIndexColIndex.MemberIndex), translationList, memberIndexColIndex.IsOrderColumn));
				}
				return new DefaultId(idArr, objContents.ObjectId.ClassId);
			}
		}

		public void InitializePSDescriptions(DatabaseDefinition db)
		{
			// intentionally left blank
		}

		public void InitializeDatabase(IDatabase db)
		{
			// intentionally left blank
		}

		public void InitializeDatabaseScript(StringCollection script)
		{
			// intentionally left blank
		}

		public IdAllocationMode GetIdAllocationMode()
		{
			return IdAllocationMode.FromAttributes;
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="keyMappers"/> is null</exception>
		public string ColumnType(ArrayList keyMappers, int columnNum)
		{
			if (keyMappers == null) throw new ArgumentNullException("keyMappers"); // Do not localize
			return (keyMappers[columnNum] as ISingleColumnAttributemapping).ColumnType(-1);
		}
		
		public void Initialize(SqlDatabaseConfig config, ArrayList keyMappers)
		{
			InternalInitialize(config, keyMappers);
		}

		private string GetStringRep(object value, ISingleColumnAttributemapping keyMapper)
		{
			string s;
			if (value is System.DateTime)
				s = ((System.DateTime)value).ToString(SqlDatabaseConfig.DateTimeFormat);
			else
				s = value.ToString();
			return keyMapper.DefaultDbValue(s, SqlDatabaseConfig);
		}

		private String GetParamStr(ObjectId id, int idIndex, int fieldIndex, bool useParams, IParameterized parameterized)
		{
			System.Object value = id.Key;
			if (useParams)
			{
				string paramName = "ID" + (KeyMappers.Count * idIndex + fieldIndex + 1).ToString(CultureInfo.InvariantCulture); // do not localize
				if (parameterized != null)
				{
					IDataParameter param = parameterized.ParamByName(paramName);
					if (KeyMappers.Count == 1)
						GetKeyMapper(0).ValueToParameter(value, param);
					else
					{
						ContentEqualArray idArr = value as ContentEqualArray;
						GetKeyMapper(fieldIndex).ValueToParameter(idArr.GetItem(fieldIndex), param);
					}
				}
				return ":" + paramName; // do not localize
			}
			else
			{
				if (KeyMappers.Count == 1)
				{
					return GetStringRep(value, GetKeyMapper(0));
				}
				else
				{
					ContentEqualArray idArr = value as ContentEqualArray;
					object keyItem = idArr.GetItem(fieldIndex);
					return GetStringRep(keyItem, GetKeyMapper(fieldIndex));
				}
			}
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="idList"/> is null</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="idColumns"/> is null</exception>
		public string IdListToWhereFragment(ObjectIdCollection idList, IParameterized parameterized, StringCollection idColumns)
		{
			if (idList == null) throw new ArgumentNullException("idList"); // Do not localize
			if (idColumns == null) throw new ArgumentNullException("idColumns"); // Do not localize
			bool useParams = idList.Count * KeyMappers.Count <= SqlDatabaseConfig.MaxParamsInIdList;

			if (KeyMappers.Count == 1)
			{
				string res = idColumns[0];
				if (idList.Count == 1)
					res += " = " + GetParamStr(idList.Any, 0, 0, useParams, parameterized);
				else
				{
				StringCollection tempList = new StringCollection();
				int idIndex = 0;
				foreach (ObjectId id in idList)
				{
					tempList.Add(GetParamStr(id, idIndex, 0, useParams, parameterized));
					idIndex++;
				}
				res += EcoSupport.SeparateStringCollection(tempList, ", ", " IN (", ")"); // do not localize
			}
			return res;
			}
			else
			{
				// composite ID
				StringBuilder sb = new StringBuilder();
				int idIndex = 0;
				sb.Append("(");
				foreach (ObjectId id in idList)
				{
					if (idIndex > 0)
						sb.Append(" OR ");
					sb.Append("(");
					for (int fieldIndex=0; fieldIndex<KeyMappers.Count; fieldIndex++)
					{
						if (fieldIndex > 0)
							sb.Append(" AND ");
						sb.Append(idColumns[fieldIndex]);
						sb.Append(" = ");
						sb.Append(GetParamStr(id, idIndex, fieldIndex, useParams, parameterized));
					}
					sb.Append(")");
					idIndex++;
				}
				sb.Append(")");
				return sb.ToString();
			}
		}
		public string IdToWhereFragment(ObjectId id, IParameterized parameterized, StringCollection idColumns)
		{
			ObjectIdCollection list = new ObjectIdCollection();
			list.Add(id);
			return IdListToWhereFragment(list, parameterized, idColumns);
		}
	}
}

